function  [gaest ka] = kappa(X)
% [gaest ka] = kappa(X) 
% Computes the estimate of the common (real) normalized fourth-order moment
% of the real and imaginary parts of the marginal variables. Also computed 
% are the (complex) normalized fourth-order moments of the marginals.  
% 
% INPUT 
% Z         = data matrix with observations as rows
% OUTPUT 
% gaest     = gamma estimate computed as eq. (10) of [1] 
% ka        = the vector of standardized 4th-order absolute moments of 
%             the marginals (c.f. eq. (6) in [1])  
% 
%  REFERENCE
% [1] Ollila and Koivunen (2009), "Adjusting the generalized likelihood ratio 
% test of circularity robust to non-normality", in Proc. 10th IEEE Int'l
% Workshop on Signal Proc.Advances in Wireless Communications (SPAWC'09), 
% Perugia, Italy, June 21-24, 2009. 
%
% (c) Esa Ollila, (esollila@wooster.hut.fi), 
% Signal Processing Laboratory, Helsinki University of Technology(c) 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


[n d] = size(X);

if n <= d  
   error('kappa: input data matrix has more variables (columns) as observations (rows)');
end

% Center to have mean zero 
Xc = X - repmat(mean(X),n,1);
Y = Xc.*conj(Xc);
vari = mean(Y,1);       % marginal variances 
pvari = mean(Xc.*Xc,1); % marginal pseudo-variances 
cc = abs(pvari./vari);  % marginal circularity coefficients 
mom  = mean(Y.^2,1);    % marginal 4th-order abs. moments 
ka = mom./(vari.^2);    % normalized 4th-order abs. moments 
gaest = 3*mean((ka./(2+cc.^2)));

