function [l p] = aglrtcirc(X,glr)
% [l p] = aglrtcirc(X,glr)
% Computes the adjusted Generalized Likelihood Ratio Test (GLRT) of circularity 
%
% The progrman computes the test statistic and the p-value of the adjusted 
% GLRT of circularity (Ollila and Koivunen, 2009). Computation of the
% p-value requires STATISTICS toolbox (MathWorks).  
%
% INPUT:
% X              =  data matrix with observations as rows. 
%                                    
% glr (optional) =  the value of the GLRT test statistic that can be 
%                   obtained using the function 'glrtcirc'.    
%
% OUTPUT 
% l                = the adjusted GLRT statistic  
% p                = p-value 
%
% USAGE 
%
% X = randn(100,3)+j*randn(100,3) % trivariate sample from circular normal
% [l p ] = aglrtcirc(X)           % value of the test stat. and p-value 
%
% REFERENCES. 
% Ollila and Koivunen (2009), "Adjusting the generalized likelihood ratio 
% test of circularity robust to non-normality", in Proc. 10th IEEE Int'l
% Workshop on Signal Proc.Advances in Wireless Communications (SPAWC'09), 
% Perugia, Italy, June 21-24, 2009. 
%
% Version 1.0, June 2009, (c) Esa Ollila (esollila@wooster.hut.fi), 
% Signal Processing Laboratory, Helsinki University of Technology
% The program was developed under Matlab 7.6.0.32
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

[n k] = size(X);

if n <= k 
    error('aglrtcirc: input data matrix has more variables (columns) as observations (rows)');
end

if nargin == 1
   glr = glrtcirc(X); % Computes the GLRT test statistic
end

gaest = kappa(X);  % Computes the kurtosis estimate 
l = 3*glr/gaest;         % Adjusting the test statistic  

if license('test','statistics_toolbox')
   p = 1 - chi2cdf(l,k*(k+1)); % p-value 
else 
   p = [ ] ;
end