function [C,invC,iter] = TylM(X,varargin)
% TylM compute the M-estimator of scatter using Huber's weight function and
% the fixed-point (FP) algorithm. 
%  
% Data is assumed to be centered (or the symmetry center parameter = 0)
%   
% USAGE
% -----
% C = TylM(X);
% [C,invC] = TylM(X,'invC',cov(X) \ eye(p),'printitn',1);
% 
% INPUT 
% -----
%   X       : the data matrix with n rows (observations) and p columns.
%             observations can be real or complex-valued. 
%
% Name-Value Pair Arguments
% -------------------------
% TylM can be called with numerous optional arguments. Optional
% arguments are given in parameter pairs, so that first argument is
% the name of the parameter and the next argument is the value for
% that parameter. Optional parameter pairs can be given in any order.
%
% Name      Value and description
%==========================================================================
%  Optional inputs: 
%   
% 'invC' :  [] (default) or a positive definite  p x p matrix
%       correspond to the inverse scatter matrix to start the FP iterations.  
%       If not specified ([]), algorithm uses the  inverse of the SCM as the
%       initial start for the FP algorithm.
%   
% 'printitn' : non-negative integer (Default is 0) 
%       Print iteration convergence. If 1, then print each iteration; if 2, 
%       then every 2nd iteration, etc. 
% 
% OUTPUT
%   C       : Tyler's M-estimator of scatter 
%   invC    : the inverse matrix of C
%   iter    : number of iterations of the FP algorithm
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

argin = inputParser;
[n, p] = size(X);

valX = @(X) assert(numel(size (X))==2 && size(X,1) >= size(X,2) && size(X,2) >= 2, ... 
        '''X'' must be a matrix having at least two columns (variables) and n >= p');
addRequired(argin,'X',valX);

if any (any (isnan (X)))
    error ('Input data contains NaN''s.');
end

if ~isa (X, 'double')
    X = double(X);
end

realdata = true;
if ~isreal(X)
   realdata=false; 
end

%-- optional input parsing rules

valinvC = @(x) assert( isempty(x) || ( ismatrix(x) && size(x,2)==size(x,1) && size(x,1)==p), ...
    'the input invC must be a empty set or matrix of size p x p');
addParameter(argin,'invC',[],valinvC);

valprintitn = @(x) assert( isscalar(x) && x >= 0 && (x==round(x)),  ... 
    '''printitn'' must be a non-negative integer');
addParameter(argin,'printitn',0, valprintitn);

%---  parse inputs
parse(argin,X,varargin{:});
invC        = argin.Results.invC;
printitn    = argin.Results.printitn; 

if p > n
    error(['data matrix X: sample size (# of rows) exceeds the data ' ...
        ' dimensionality (# of columns)']);
end

if isempty(invC) 
    C = X'*X/n;
    C = p*C/trace(C);
    invC = C\eye(p);
end
           
MAX_ITER = 5000; % Max number of iteration
EPS = 5.0e-4;    % Iteration accuracy
iter = 1;

while (iter<MAX_ITER)   
    
      t = real(sum((X*invC).*conj(X),2)); % norms        
      C = (p/n)*X'*(X.*repmat(1./t,1,p)); 
      C = p*C/trace(C);
      err = eye(p)-invC*C;
      d = norm(err(:),Inf);
      
      if mod(iter,printitn)==0
            fprintf('At iter = %4d, dis=%.6f\n',iter,d);
      end
      
      invC = C\eye(p);
    
      if (d<=EPS) 
         break;             
      end         
           
      iter = iter+1;  
end

C(1:(p+1):p^2) = real(C(1:(p+1):p^2)); 

if (iter==MAX_ITER)
     error(['WARNING! Slow convergence: error of the solution is %f\n ' ...
         'after %d iterations\n'],d,iter);
end
