function [Cest,beta0,C,stats,invC] = RTylM(X,varargin)
% RTylM computes the regularized (shrinkage) M-estimator of scatter of [1] 
% based on Tyler's weight function and the fixed-point (FP) algorithm. 
% 
% Data is assumed to be centered (or the symmetry center parameter = 0). If
% your data is not centered, then please center your data. 
%
% USAGE (examples)
% ----------------
% Cest = RTylM(X);
% [Cest,beta0] = RTylM(X,'printitn',1);
% 
% INPUT 
% -----
%   X       : the data matrix with n rows (observations) and p columns.
%             observations can be real or complex-valued. 
%
% Name-Value Pair Arguments
% -------------------------
% RTylM can be called with numerous optional arguments. Optional
% arguments are given in parameter pairs, so that first argument is
% the name of the parameter and the next argument is the value for
% that parameter. Optional parameter pairs can be given in any order.
%
% Name      Value and description
%==========================================================================
% --Basic parameter is the choise of the estimator to use
%
% 'approach' : (string) defines which method is used to estimate the 
%       shrinkage parameter beta. Valid choises are 
%           'ell1' (default)   uses RTYL-ELL1 estimator
%           'ell2'             uses RTYL-ELL2 estimator
%
% -- other optional inputs: 
% 
% 'invC' :  [] (default) or a positive definite  p x p matrix
%       correspond to the inverse scatter matrix to start the FP iterations.  
%       If not specified ([]), algorithm uses the  inverse of the SCM as the
%       initial start for the FP algorithm.
%   
% 'printitn' : non-negative integer (Default is 0) 
%       Print iteration convergence. If 1, then print each iteration; if 2, 
%       then every 2nd iteration, etc. 
%   
% 'gamma' : [] (default) or real number between [1,p] 
%       Sphericity measure to use.  If not given ([]), then the function 
%       computes the Ell1-estimator of sphericity. 
%  
% OUTPUT
% ------
%   Cest    : the shrinkage M-estimator of scatter using Tyler's weight 
%   beta0   : the found optimal MMSE shrinkage parameter beta
%   C       : Tyler's conventional (non-shrinked) M-estimator
%   stats   : a structure array with fields
%       .psi1     : computed estimate of psi1
%       .eta      : trace(C)/p 
%       .iter     : nr of FP iterations
%       .gamest   : Ell1 or Ell2-estimate of sphericity
%       .approach : approach used 
%   invC    : the inverse of C
%
% REFERENCE 
% ---------
% [1] E. Ollila, D.P. Palomar, and F. Pascal, "Shrinking the eigenvalues of 
%  M-estimators of covariance matrix", Arxiv, 2020. 
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

argin = inputParser;
[n, p] = size(X);

valX = @(X) assert(numel(size (X))==2 && size(X,1) >= size(X,2) && size(X,2) >= 2, ... 
        '''X'' must be a matrix having at least two columns (variables) and n >? p ');
addRequired(argin,'X',valX);

if any (any (isnan (X)))
    error ('Input data contains NaN''s.');
end

if ~isa (X, 'double')
    X = double(X);
end

realdata = true;
if ~isreal(X)
   realdata=false; 
end

%-- optional input parsing rules
valinvC = @(x) assert( isempty(x) || ( ismatrix(x) && size(x,2)==size(x,1) && size(x,1)==p), ...
    'the input invC must be a empty set or matrix of size p x p');
addParameter(argin,'invC',[],valinvC);

valprintitn = @(x) assert( isscalar(x) && x >= 0 && (x==round(x)),  ... 
    '''printitn'' must be a non-negative integer');
addParameter(argin,'printitn',0, valprintitn);

valGamma = @(x) assert(isreal(x) && x >= 1 && x < p,['''gamma'' must be ' ...
    'real-valued and in the range [1,' num2str(p) ')']);
addParameter(argin,'gamma',[], valGamma);

addParameter(argin,'approach','ell1', ... 
    @(x) logical(sum(strcmp(x,{'ell1','ell2'}))));
% must be ell1 or ell2 

%---  parse inputs
parse(argin,X,varargin{:});
invC        = argin.Results.invC;
printitn    = argin.Results.printitn; 
compute_gamma = isempty(argin.Results.gamma);
approach = argin.Results.approach;

%% 
[C,invC,iter] = TylM(X,'invC',invC,'printitn',printitn);

%-- compute psi1
if isreal(X) 
    psi1 = p/(p+2);
else
    psi1 = p/(p+1);
end

%-- Compute the sphericity 
if compute_gamma
    switch approach 
        case 'ell1'
            gamest = gamell1(X,true); 
        case 'ell2'
             gamest0 = trace(C^2)/p;
             gamest = gamell2M(n,p,psi1,gamest0,realdata);
        otherwise
            % Hmmm, something wrong with the parameter string
            error(['Unrecognized parameter: ''' approach '''']);           
    end
else 
        gamest = argin.Results.gamma;
end
gam0 = gamest - 1; % translate gamest, so that min value is 0.

if isreal(X) 
    psi1 = p/(p+2);
    beta0 = gam0/( gam0*(1-1/n) + psi1*(1-1/p)*(2*gamest + p)/n);
else
    psi1 = p/(p+1);
    beta0 = gam0/( gam0*(1-1/n) + psi1*(1-1/p)*(gamest + p)/n);
end

Cest = beta0*C + (1-beta0)*eye(p);

stats.psi1 = psi1;
stats.iter = iter;
stats.gamest = gamest;
stats.approach = approach;

