function [Cest,beta0,C,stats,invC] = RMVT(X,varargin)
% RMVT computes the shrinkage M-estimator of scatter of [1] using 
% multivariate t-distribution (MVT) weight function and the fixed-point 
% (FP) algorithm. The degrees of freedom (d.o.f.) parameter *nu* of 
% t-distribution is adaptively estimated from the data unless it is given a
% prespecified value. 
% 
% If the data follows a MVT distribution with the correctly determined 
% d.o.f., then the underlying M-estimator is also the maximum likelihood 
% estimate (MLE) of the scatter matrix parameter. 
%
% Data is assumed to be centered (or the symmetry center parameter = 0). 
%                 
% INPUT 
% -----
%   X       : the data matrix with n rows (observations) and p columns.
%             observations can be real or complex-valued. 
%
% Name-Value Pair Arguments
% -------------------------
% RMVT can be called with numerous optional arguments. Optional
% arguments are given in parameter pairs, so that first argument is
% the name of the parameter and the next argument is the value for
% that parameter. Optional parameter pairs can be given in any order.
%
% Name      Value and description
%==========================================================================
% --Basic parameter is the choise of the estimator to use
%
% 'approach' : (string) defines which method is used to estimate the 
%       shrinkage parameter beta. Valid choises are 
%           'ell1' (default)   uses RMVT-ELL1 estimator
%           'ell2'             uses RMVT-ELL2 estimator
%
% -- other optional inputs: 
% 
%   'nu' : non-negative real scalar or empty set ([])
%       correponds to d.o.f. parameter  of the t-distribution. 
%       If not given (default = []), then  nu is estimated  
%       using Algorithm 1 described in [1]. 
%       - nu = Inf implies using Gaussian weight u(t) = t in which case
%       the function returns the sample covariance matrix
%       - nu > 0 implies using a t-weight function u(t;nu) 
%       
%   'invC' : [] (default) or a pos. def. p x p matrix
%       correspond to the inverse scatter matrix to start the FP iterations.  
%       If not specified ([]), algorithm uses the  inverse of the SCM as 
%       initial start for the FP algorithm.
%
%  'scaling' : 'none' (default), 'gaussian', 'covariance'
%       -'none' implies no scaling, so using the standard MVT weight 
%           function u(t;v) = (p+v)/(v+t)
%       -'covariance' implies using weight function corresponding to MVT 
%           distribution where the scatter matrix coincides with the 
%           covariance matrix, so u(t;v)= (p+v)/(v-2+t). Requires nu > 2. 
%       -'gaussian' implies using MVT weight function u(t;nu) that is s
%           scaled so that the estimator is consistent to covariance matrix 
%           for Gaussian  data 
%
% 'iter_nu' : positive integer (Default is 2) 
%       maximum number of iterations used to estimate the nu parameter.  
%
% 'printitn' : non-negative integer (Default is 0) 
%       Print iteration convergence. If 1, then print each iteration, if 2, 
%       then every 2nd iteration, etc. 
%
% 'gamma' : [] (default) or real number between [1,p] 
%       Sphericity measure to use.  If not given ([]), then the function 
%       computes the Ell1-estimator of sphericity. 
%  
% OUTPUT
% ------
%   Cest    : the shrinkage M-estimator of scatter using MVT weight 
%   beta    : the found shrinkage parameter beta
%   C       : the conventional (non-shrinked) M-estimator (using beta=1)
%   stats   : a structure array with fields
%       .psi1    : computed estimate of psi1
%       .eta     : trace(C)/p 
%       .iter    : nr of FP iterations
%       .gamest  : Ell1-estimate of sphericity
%       .nu      : the d.o.f. parameter of MVT weight
%   invC    : the inverse of C
%
% REFERENCE 
% ---------
% [1] E. Ollila, D.P. Palomar, and F. Pascal, "Shrinking the eigenvalues of 
%       M-estimators of covariance matrix", Arxiv, 2020. 
%
% Author: Esa Ollila, May 2020, Aalto University. 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

[n, p] = size(X);
argin = inputParser;

valX = @(X) assert(numel(size(X))==2 && size(X,1) >= size(X,2) && size(X,2) >= 2, ... 
        '''X'' must be a matrix having at least two columns (variables) and n >= p');
addRequired(argin,'X',valX);

if any(any(isnan(X)))
    error ('Input data contains NaN''s.');
end

if ~isa(X,'double')
    X = double(X);
end

realdata = true;
if ~isreal(X)
   realdata=false; 
end

%-- optional input parsing rules
valnu = @(x) assert(isempty(x) || (isscalar(x) && x > 0), ... 
    '''nu'' must be positive scalar or empty set');
addParameter(argin,'nu',[], valnu);

valinvC = @(x) assert( isempty(x) || ( ismatrix(x) && size(x,2)==size(x,1) && size(x,1)==p), ...
    'the input invC must be a empty set or matrix of size p x p');
addParameter(argin,'invC',[],valinvC);

valscaling = @(x) assert(any(strcmpi(x,{'gaussian','covariance','none'})), ...
    'must be string equal to ''gaussian'' or ''covariance'' or ''none''');
addParameter(argin,'scaling','none',valscaling);

valprintitn = @(x) assert( isscalar(x) && x >= 0 && (x==round(x)),  ... 
    '''printitn'' must be a non-negative integer');
addParameter(argin,'printitn',0, valprintitn);

valiter = @(x) assert(isempty(x) || ((isscalar(x) && x > 0) && round(x)==x), ... 
    '''iter_nu'' must be positive integer');
addParameter(argin,'iter_nu',2, valiter);

valGamma = @(x) assert( isempty(x) || (isreal(x) && x >= 1 && x < p), ... 
    ['''gamma'' must be empty set or real-valued scalar in the range [1,' num2str(p) ')']);
addParameter(argin,'gamma',[], valGamma);

addParameter(argin,'approach','ell1', ... 
    @(x) logical(sum(strcmp(x,{'ell1','ell2'}))));
% must be ell1 or ell2 

%---  parse inputs
parse(argin,X,varargin{:});
scaling     = argin.Results.scaling;
invC1       = argin.Results.invC;
nu          = argin.Results.nu;
printitn    = argin.Results.printitn; 
MAX_ITER_nu = argin.Results.iter_nu;
compute_gamma = isempty(argin.Results.gamma);
approach = argin.Results.approach;

%-- Compute the t-MLE 
[C,invC,nu,b,iter] = MVT(X,'nu',nu,'invC',invC1,'scaling',scaling, ... 
    'printitn',printitn,'iter_nu',MAX_ITER_nu);
eta = trace(C)/p;
        
if isinf(nu)
% if the nu estimate is infinity 
      psi1 = 1;
else
% by default, the scaling constant is b=1    
      if realdata
          psi1 = (1/b^2)*((p+nu)/(p+nu+2));
      else
          psi1 = (1/b^2)*((2*p+nu)/(2*p+nu+2));
      end
end

%-- Compute the sphericity 
if compute_gamma
    switch approach 
        case 'ell1'
            gamest = gamell1(X,true); 
        case 'ell2'
             gamest0 =  (trace(C^2)/p)/eta^2;
             gamest = gamell2M(n,p,psi1,gamest0,realdata);
        otherwise
            % Hmmm, something wrong with the parameter string
            error(['Unrecognized parameter: ''' approach '''']);           
    end
else 
        gamest = argin.Results.gamma;
end
T = gamest - 1;

%-- compute beta 
if realdata
    beta0 = T/( T*(1-1/n) + psi1*(1-1/p)*(2*gamest + p)/n);
else
    beta0 = T/( T*(1-1/n) + psi1*(1-1/p)*(gamest + p)/n);
end

% assure  that beta is between 0 and 1 
beta0 = min(max(0,beta0),1); 

%-- compute the Regularized MVT MLE:
Cest = beta0*C + (1-beta0)*eta*eye(p);

stats.beta = beta0;
stats.psi1 = psi1;
stats.eta = eta;
stats.iter = iter;
stats.gamest = gamest;
stats.nu = nu;
stats.approach = approach;




