function [Cest,beta0,C,stats,invC] = RHubM(X,varargin)
% RHubM computes the regularized (shrinkage) M-estimator of scatter of [1] 
% based on Huber's weight function and the fixed-point (FP) algorithm. 
% 
% Data is assumed to be centered (or the symmetry center parameter = 0). If
% your data is not centered, then please center your data. 
%
% USAGE
% -----
% Cest = RHubM(X);
% [Cest,beta0] = RHubM(X,'q',0.7,'invC',cov(X) \ eye(p),'printitn',1);
% 
% INPUT 
% -----
%   X       : the data matrix with n rows (observations) and p columns.
%             observations can be real or complex-valued. 
%
% Name-Value Pair Arguments
% -------------------------
% RHubM can be called with numerous optional arguments. Optional
% arguments are given in parameter pairs, so that first argument is
% the name of the parameter and the next argument is the value for
% that parameter. Optional parameter pairs can be given in any order.
%
% Name      Value and description
%==========================================================================
% --Basic parameter is the choise of the estimator to use
%
% 'approach' : (string) defines which method is used to estimate the 
%       shrinkage parameter beta. Valid choises are 
%           'ell1' (default)   uses RHUB-ELL1 estimator
%           'ell2'             uses RHUB-ELL2 estimator
%
% -- other optional inputs: 
% 
% 'q' : real scalar in (0,1); default q=0.9
%       determines the threshold c^2 of Huber's weight function.  
%   
% 'invC' :  [] (default) or a positive definite  p x p matrix
%       correspond to the inverse scatter matrix to start the FP iterations.  
%       If not specified ([]), algorithm uses the  inverse of the SCM as the
%       initial start for the FP algorithm.
%   
% 'printitn' : non-negative integer (Default is 0) 
%       Print iteration convergence. If 1, then print each iteration; if 2, 
%       then every 2nd iteration, etc. 
%   
% 'gamma' : [] (default) or real number between [1,p] 
%       Sphericity measure to use.  If not given ([]), then the function 
%       computes the Ell1-estimator of sphericity. 
%  
% OUTPUT
% ------
%   Cest    : the shrinkage M-estimator of scatter using Huber's weight 
%   beta0   : the found optimal MMSE shrinkage parameter beta
%   C       : Huber's conventional (non-shrinked) M-estimator
%   stats   : a structure array with fields
%       .psi1    : computed estimate of psi1
%       .eta     : trace(C)/p 
%       .gamest  : Ell1-estimate of sphericity
%       .iter    : nr of FP iterations
%       .q       : the used quantile 
%       .csq     : threshold c^2 of Huber's weight (depends on q)
%       .b       : scaling factor of Huber's weight function (depends on q)
%       .t       : Mahalanobis distances based on Huber's M-estimator C
%   invC    : the inverse of C
%
% REFERENCE 
% ---------
% [1] E. Ollila, D.P. Palomar, and F. Pascal, "Shrinking the eigenvalues of 
%  M-estimators of covariance matrix", Arxiv, 2020. 
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

argin = inputParser;
[n, p] = size(X);

%- Huber's function (without the scaling by 1/b):
ufun = @(t,c) ((t<=c) + (c./t).*(t>c)); % weight function u(t)

valX = @(X) assert(numel(size (X))==2 && size(X,1) >= size(X,2) && size(X,2) >= 2, ... 
        '''X'' must be a matrix having at least two columns (variables) and n >= p ');
addRequired(argin,'X',valX);

if any (any (isnan (X)))
    error ('Input data contains NaN''s.');
end

if ~isa (X, 'double')
    X = double(X);
end

realdata = true;
if ~isreal(X)
   realdata=false; 
end

%-- optional input parsing rules
valq = @(x) assert(isscalar(x) && (x > 0) && (x <1),'''q'' must be a scalar in (0,1)');
addParameter(argin,'q',0.9, valq);

valinvC = @(x) assert( isempty(x) || ( ismatrix(x) && size(x,2)==size(x,1) && size(x,1)==p), ...
    'the input invC must be a empty set or matrix of size p x p');
addParameter(argin,'invC',[],valinvC);

valprintitn = @(x) assert( isscalar(x) && x >= 0 && (x==round(x)),  ... 
    '''printitn'' must be a non-negative integer');
addParameter(argin,'printitn',0, valprintitn);

valGamma = @(x) assert(isreal(x) && x >= 1 && x < p,['''gamma'' must be ' ...
    'real-valued and in the range [1,' num2str(p) ')']);
addParameter(argin,'gamma',[], valGamma);

addParameter(argin,'approach','ell1', ... 
    @(x) logical(sum(strcmp(x,{'ell1','ell2'}))));
% must be ell1 or ell2 
%---  parse inputs
parse(argin,X,varargin{:});
invC        = argin.Results.invC;
q           = argin.Results.q;
printitn    = argin.Results.printitn; 
compute_gamma = isempty(argin.Results.gamma);
approach = argin.Results.approach;

%% Compute the Huber's M-estimator of scatter  and the scale \eta:
[C,invC,csq,b,t,iter] = HubM(X,'q',q,'invC',invC,'printitn',printitn);
eta = trace(C)/p; % scale (mean of the eigenvalues)

%% Compute the psi_1 parameter:
if n >= (1.5)*p
    Xw = winsorize(X,t,csq,b);
    psi1 = 1+ellkurt(Xw,zeros(p,1),true);
else
    psi1 = mean(((1/b)*ufun(t,csq)).^2.*(t.^2))/(p*(p+2));
end

%% Compute the sphericity 
if compute_gamma
    switch approach 
        case 'ell1'
            gamest = gamell1(X,true); 
        case 'ell2'
             gamest0 =  (trace(C^2)/p)/eta^2;
             gamest = gamell2M(n,p,psi1,gamest0,realdata);
        otherwise
            % Hmmm, something wrong with the parameter string
            error(['Unrecognized parameter: ''' approach '''']);           
    end
else 
        gamest = argin.Results.gamma;
end
gam0 = gamest - 1; % translate gamest, so that min value is 0.


%% Compute the shrinkage parameter \beta
if realdata 
    beta0 = gam0/( gam0*(1-1/n) + psi1*(1-1/p)*(2*gamest + p)/n);
else
    beta0 = gam0/( gam0*(1-1/n) + psi1*(1-1/p)*(gamest + p)/n);
end 
% make sure that beta value  is between 0 and 1 
beta0 = min(max(0,beta0),1);

%% Compute the shrinkage Huber's M-estimator RHub-Ell1
Cest = beta0*C + (1-beta0)*eta*eye(p);

stats.psi1 = psi1;
stats.eta = eta;
stats.gamest = gamest;
stats.iter = iter;
stats.q = q;
stats.csq = csq;
stats.t = t;
stats.approach = approach;

end
%% Winsorize function
function Xw = winsorize(X,t,csq,b)
% Xw = winsorise(X,t,csq,b)
% Computes the winsorized data Xw = (1/sqrt(b))*wins(X), where 
%           x,                              if  ||C^(-1/2) x ||^2  <= csq
% wins(x) =  
%           sqrt(csq) x /|| C^(-1/2) x ||,  if  ||C^(-1/2) x ||^2  > csq
% 
% Input:
%   X       n x p (real or complex) matrix. Data matrix with n observations 
%           as rows and  p variables as columns. 
%   t       n x 1 vector of positive reals (optional input).  Consists
%           of suared Mahalanobis distances t(i)= || invC^(1/2)*X(i,:)||^2
%   csq     winsorize observations for which t > csq 
%   b       scaling factor
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

Xw = (1/sqrt(b))*X;
if any(t>csq)
    Xw(t>csq,:)= sqrt(csq/b)*X(t>csq,:)./sqrt(t(t>csq));
end
end




