function [C,invC,csq,b,t,iter] = HubM(X,varargin)
% HubM compute the M-estimator of scatter using Huber's weight function and
% the fixed-point (FP) algorithm. 
%  
% Data is assumed to be centered (or the symmetry center parameter = 0)
%   
% USAGE
% -----
% C = HubM(X);
% [C,invC] = HubM(X,'q',0.7,'invC',cov(X) \ eye(p),'printitn',1);
% 
% INPUT 
% -----
%   X       : the data matrix with n rows (observations) and p columns.
%             observations can be real or complex-valued. 
%
% Name-Value Pair Arguments
% -------------------------
% HubM can be called with numerous optional arguments. Optional
% arguments are given in parameter pairs, so that first argument is
% the name of the parameter and the next argument is the value for
% that parameter. Optional parameter pairs can be given in any order.
%
% Name      Value and description
%==========================================================================
%  Optional inputs: 
% 
%   'q'     (real scalar in (0,1))
%   Upper quantile  of Chi-squared distribution F with p degrees of
%   freedom. In real-valued case, it determines the threshold of 
%   Huber's function  via c^2 = F^-1(q).
%   
%   'invC'  (pos. def. p x p matrix)
%   The inverse scatter matrix to start the FP iterations. If not given 
%   (invC=[] by default), then compute invC as the inverse of the SCM.
%   
%   'printitn'  (non-negative integer). 
%    Print iteration convegence (default is 0, which implies do not print, 
%    1 means print each iteration, etc)
%
% OUTPUT
%   C       : M-estimator of scatter for t-weight u(t;nu)
%   invC    : the inverse matrix of C
%   csq     : tuning constant of Huber's weight function (depends on q)
%   b       : scaling factor of Huber's weight function (depends on q)
%   t       : squared Mahalanobis distances x'*invC*x for the data
%   iter    : number of iterations of the FP algorithm
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


argin = inputParser;
[n, p] = size(X);

valX = @(X) assert(numel(size (X))==2 && size(X,1) >= size(X,2) && size(X,2) >= 2, ... 
        '''X'' must be a matrix having at least two columns (variables) and n >= p');
addRequired(argin,'X',valX);

if any (any (isnan (X)))
    error ('Input data contains NaN''s.');
end

if ~isa (X, 'double')
    X = double(X);
end

realdata = true;
if ~isreal(X)
   realdata=false; 
end

%-- optional input parsing rules
valq = @(x) assert(isscalar(x) && (x > 0) && (x <1),'''q'' must be a scalar in (0,1)');
addParameter(argin,'q',0.9, valq);

valinvC = @(x) assert( isempty(x) || ( ismatrix(x) && size(x,2)==size(x,1) && size(x,1)==p), ...
    'the input invC must be a empty set or matrix of size p x p');
addParameter(argin,'invC',[],valinvC);

valprintitn = @(x) assert( isscalar(x) && x >= 0 && (x==round(x)),  ... 
    '''printitn'' must be a non-negative integer');
addParameter(argin,'printitn',0, valprintitn);

%---  parse inputs
parse(argin,X,varargin{:});
invC        = argin.Results.invC;
q           = argin.Results.q;
printitn    = argin.Results.printitn; 

if isempty(invC) 
    C = X'*X/n;
    invC = C\eye(p);
end

%- Huber's function (without the scaling by 1/b):
ufun = @(t,c) ((t<=c) + (c./t).*(t>c)); % weight function u(t)

%-- compute c^2 = F_{\Chi^2_p}(q) and scaling factor b of the weight fnc
if realdata
   csq = chi2inv(q,p); % threshold used in Huber's weight u(t;.)
   b = chi2cdf(csq,p+2)+(csq/p)*(1-q); % consistency factor
else
   csq = chi2inv(q,2*p)/2;  
   b = chi2cdf(2*csq,2*(p+1))+(csq/p)*(1-q); 
end
      
const = (1/(b*n));

MAX_ITER = 2500; % Max number of iteration
EPS = 5.0e-4;    % Iteration accuracy
iter = 1;

while (iter<=MAX_ITER)   
    
      t = real(sum((X*invC).*conj(X),2)); % norms        
      C = const*X'*(X.*repmat(ufun(t,csq),1,p)); 
      err = eye(p)-invC*C;
      d = norm(err(:),Inf);
      
      if mod(iter,printitn)==0
        fprintf('At iter = %4d, dis=%.6f\n',iter,d);
      end
      
      invC = C\eye(p);
    
      if (d<=EPS) 
         break;             
      end         
           
      iter = iter+1;  
end

t = real(sum((X*invC).*conj(X),2)); % norms        
C(1:(p+1):p^2) = real(C(1:(p+1):p^2)); 

if (iter==MAX_ITER)
     error(['WARNING! Slow convergence: error of the solution is %f\n ' ...
         'after %d iterations\n'],d,iter);
end
