function  [yhat,stat] = RegQDA(Xt,X,y,mu,appr,pr,kap) 
% RegQDA computes misclassifaction rates of regularized QDA 
% methods based on regularized SCM estimators (Ell1-, Ell2, 
% or LW-)RSCM. 
%
% inputs:
%   Xt   Nt x p test data matrix consisting of observations from K classes
%   X    N x p data matrix where observations in each class are already 
%        assumed to be centered w.r.t. to their class sample means! Hence 
%        it holds that mean(X(y==ii,:))  should be close to zero vector for
%        all ii =1,...,K.
%   y    N x 1 matrix of class labels
%   app  string equal to 'ell2', 'ell1', or 'lw'
%
% optional inputs:
%   pr   optional class a priori probabilities (a vector of size 1xK)
%        Elements of pr are in [0,1] and should sum to 1. Defaul value 
%        is pr = (1/K, ..., 1/K). 
%   kap  vector of elliptical kurtosis values for each class. 
%
% usage: 
%   err = RegQDA(Xt,X,y,mu,app),
%   [err,stat] = RegQDA(Xt,X,y,mu,appr,pr,kap);
%
% toolbox: RegularizedSCM ({esa.ollila,elias.raninen}@aalto.fi)
%--------------------------------------------------------------------------


p = size(X,2);
K = max(y);
Nt = size(Xt,1);
stat  = cell(K,1); 
dqda = zeros(Nt,K);
n = histcounts(y);

if norm(mean(X(y==1,:))) > 10^-7
    error(['RegQDA: training data is assumed to be centered w.r.t. the ' ...
            ' class means']);
end

if  nargin < 6 
    pr = ones(1,K);
end

for ii=1:K  
    if nargin == 7
         [~,iRSCM, stat{ii}] = regscm(X(y==ii,:),'approach',appr,'inverse',true, ... 
             'verbose','off','centered',true,'kappa',kap(ii));
    else
         [~,iRSCM, stat{ii}] = regscm(X(y==ii,:),'approach',appr,'inverse',true, ... 
             'verbose','off','centered',true);
    end
    if p/n(ii) >= 10 
        a = sum(log(stat{ii}.eigval));
    else
        a = -log(det(iRSCM));
    end
    Xt_c = Xt - repmat(mu(:,ii).',Nt,1); % centered test data
    dqda(:,ii) = sum((Xt_c*iRSCM).*Xt_c,2) + a - 2*log(pr(ii));   
end

[~,yhat]=min(dqda,[],2);
