/*
 rcmreg.c -- Esa Ollila.  

 content: R callable C-function for computation of the regression 
 coefficients based on the lift rank covariance matrix. The method 
 is described in:

 E. Ollila, H. Oja and V. Koivunen : "Estimates of regression 
 coefficients based on lift rank covariance matrix," Journal 
 of the American Statistical Association, vol. 98, no. 461, 
 pp. 90 -- 98, 2003. 

 author:  Esa Ollila, esollila@wooster.hut.fi
          http://cc.oulu.fi/~esollila/

 This C-code is a public domain, altough please leave my name, 
 email address as well as the web address attached.  If you use 
 this program or any subroutines in it, please cite the above mentioned 
 refrence  

 To compile this program in UNIX use : 

 R CMD rcmreg.c nr.c -o extojarnk.so
 
 Then in R you can you use the routine rcmreg.R as follows: 

 rcmreg(data,dx) 

 where 'data' is your data set with n rows and k columns 
 and 'dx' is the number of explanatory variables. 
 The explanatory variables should be the last dx columns in 'data'
*/
#include <stdlib.h>
#include <math.h>
#include "rcmreg.h"
#include <R.h>

#define AUTHOR Esa Ollila
#define EMAIL esollila@wooster.hut.fi
#define WWW "http://cc.oulu.fi/~esollila/"

double **data, **data2, **score; 
int k,n;     

void extojarnk(double *data2, double *score2, long *n2, long *k2)
{
        int i, j;
	
	n = (int)*n2;
	k = (int)*k2;

       /* Allocate memory for global variables */
	data  = (double**)R_alloc(n+1,sizeof(double*));
	score = (double**)R_alloc(n+1,sizeof(double*));
	for (i=0;i<=n;i++) data[i]  = (double*)R_alloc(k+1,sizeof(double));
	for (i=0;i<=n;i++) score[i] = (double*)R_alloc(k+2,sizeof(double));

	for (j=1;j<=k;j++)
	    for (i=1;i<=n;i++)
 	        data[i][j] = data2[(j-1)*n+(i-1)]; 

	Oja_Ranks(); 

	for (j=0;j<=k;j++)
  	    for (i=1;i<=n;i++)
 	        score2[j*n+(i-1)] = score[i][j+1]; 
}

void Oja_Ranks()
/* main function for computation of lift Oja ranks */
{   
        int i,j,k1,*p,*indx,lst,sp;   
	unsigned long cnt;	
	double d,**dmat,*dp,*h;
	long idum;

	p = (int*)R_alloc(k+1,sizeof(int)); 
	h = (double*)R_alloc(k+1,sizeof(double)); 
	indx = (int*)R_alloc(k+2,sizeof(int));
	dmat = (double**)R_alloc(k+2,sizeof(double*)); 
	dp = (double*)R_alloc(k+2,sizeof(double)); 
	for (i=0;i<=k+1;i++) dmat[i]  = (double*)R_alloc(k+2,sizeof(double));

	i=0;
	do {  
	    d  = (double)(rand()%256);
	    if (d!=0)  h[++i] = d/256;
	} while (i<k);

	k1=k+1;       
	for (i=1;i<=n;i++) 
	    for (j=1;j<=k1;j++)
	        score[i][j] = 0;      
      
        for (j=1;j<=k;j++) p[j] = j;
        cnt= 0;	
	do {
	    cnt += 1;
	    for (i=1;i<=k1;i++) dmat[i][1] = 1;
	    for (i=1;i<=k;i++) 
		for (j=1;j<=k;j++) 
		    dmat[j][i+1] = data[p[j]][i];
	    for (i=1;i<=k;i++) dmat[k1][i+1] = h[i];
	    /* cofactors */
	    j = ludcmp(dmat,k1,indx,&d);  	   
	    for (i=1;i<=k1;i++) d *= dmat[i][i]; 	
	    for (i=1,dp[k1]=1;i<=k;i++) dp[i]=0;	
	    lubksb(dmat,k1,indx,dp);	
	    for (i=1;i<=k1;i++) dp[i] *= d;
	    /* cumulate ranks */
	    for (i=1;i<=n;i++) {	       
		for (j=1;j<=k;j++) {
		    if (i==p[j]) break;
		}	   
		if (j==k1) { 		       	
		   for (j=1,d=dp[1];j<=k;j++) d += data[i][j]*dp[j+1];		
		   if (fabs(d)>SMALL) {	       
		      sp = SIGN(d);		    		  
		      for (j=1;j<=k1;j++) score[i][j] += sp*dp[j];	           
		   }
	        }
	    }
	    Nextp(n,k,p,&lst);
	    if (lst==0) break;	    
        } while(1);

        for (i=1;i<=n;i++) 
	    for (j=1;j<=k1;j++) 
		score[i][j] = score[i][j]/cnt;			  	      
}      
  
void Nextp(int n, int k, int *q, int *lc)
/* next subset selection */ 
{
	int i;

	*lc = k;
	if (q[k]<n) {
		q[k] +=1;
	} else {
		do {
			*lc = *lc-1;
			if ( q[*lc]< (n-k+(*lc))) break;
		} while ( (*lc) !=0 );

		if ((*lc)>0) {
			q[(*lc)] += 1;
			for (i=(*lc)+1;i<=k;i++)
				q[i] = q[i-1]+1;
		}
	}
}











