/*    
  These functions copyrighted property and are explained in the  book:
      
        Press, Teukolsky, Vetterling, Flannery : Numerical recipes in C.
        2. edition. Cambidge University Press.   
*/
#include <math.h>
#include "rcmreg.h"
#include <R.h>
#define TINY 1.0e-20; /* A small number. */

int ludcmp(double **a, int n, int *indx, double *d)
{
	int i,imax,j,k;
	double big,dum,sum,temp;
	double *vv; /*vv stores the implicit scaling of each row.*/

	vv=(double*)R_alloc(n+1,sizeof(double));
	*d=1.0; /* No row interchanges yet. */
	for (i=1;i<=n;i++) { /* Loop over rows to get the implicit scaling information. */
		big=0.0;
		for (j=1;j<=n;j++)
		if ((temp=fabs(a[i][j])) > big) big=temp;
		if (big == 0.0) {
			 /*printf("Singular matrix in routine ludcmp");*/
			return -1;
		}
			 
		/*No nonzero largest element.*/
		vv[i]=1.0/big; /*Save the scaling.*/
	}
	for (j=1;j<=n;j++) { /*This is the loop over columns of Crout's method.*/
		for (i=1;i<j;i++) { /*This is equation (2.3.12) except for i = j.*/
			sum=a[i][j];
			for (k=1;k<i;k++) sum -= a[i][k]*a[k][j];
			a[i][j]=sum;
		}
		big=0.0; /*Initialize for the search for largest pivot element. */
		for (i=j;i<=n;i++) { /* This is i = j of equation (2.3.12) and i = j +1, ..,N */
			sum=a[i][j];     /* of equation (2.3.13). */
			for (k=1;k<j;k++)
				sum -= a[i][k]*a[k][j];
			a[i][j]=sum;
			if ( (dum=vv[i]*fabs(sum)) >= big) {
			/* Is the figure of merit for the pivot better than the best so far? */
				big=dum;
				imax=i;
			}
		}
		if (j != imax) { /* Do we need to interchange rows? */
			for (k=1;k<=n;k++) {  /* Yes, do so... */
				dum=a[imax][k];
				a[imax][k]=a[j][k];
				a[j][k]=dum;
			}
			*d = -(*d); /* ...and change the parity of d. */
			vv[imax]=vv[j]; /* Also interchange the scale factor. */
		}
                 indx[j]=imax;
		if (a[j][j] == 0.0) a[j][j]=TINY;
		if (j != n) { /* Now, finally, divide by the pivot element. */
			dum=1.0/(a[j][j]);
			for (i=j+1;i<=n;i++) a[i][j] *= dum;
		}
	}  /* Go back for the next column in the reduction. */    
	
	return 0;
}

void lubksb(double **a, int n, int *indx, double b[])
/* Solves the set of n linear equations A*X = B. Here a[1..n][1..n] is input, not as the matrix
A but rather as its LU decomposition, determined by the routine ludcmp. indx[1..n] is input
as the permutation vector returned by ludcmp. b[1..n] is input as the right-hand side vector
B, and returns with the solution vector X. a, n, and indx are not modiffed by this routine
and can be left in place for successive calls with different right-hand sides b. This routine takes
into account the possibility that b will begin with many zero elements, so it is efficient for use
in matrix inversion. */
{
	int i,ii=0,ip,j;
	double sum;
	for (i=1;i<=n;i++) { 
		ip=indx[i];
		sum=b[ip];
		b[ip]=b[i];
		if (ii)
			for (j=ii;j<=i-1;j++) sum -= a[i][j]*b[j];
		else if (sum) ii=i; /* A nonzero element was encountered, so from now on we
				       will have to do the sums in the loop above. */
		b[i]=sum; 
	}
	for (i=n;i>=1;i--) { /* Now we do the backsubstitution, equation (2.3.7). */
		sum=b[i];
		for (j=i+1;j<=n;j++) sum -= a[i][j]*b[j];
		b[i]=sum/a[i][i]; /* Store a component of the solution vector X. */
	} /* All done! */
}
