/*    
  These functions are copyrighted property and are explained in the book:
      
        Press, Teukolsky, Vetterling, Flannery : Numerical recipes in C.
        2. edition. Cambidge University Press.   
*/
#include <math.h>
#include <stdio.h>
#include "rcmreg.h"
#define TINY 1.0e-20; /* A small number. */
#define IA 16807 
#define IM 2147483647 
#define AM (1.0/IM) 
#define IQ 127773 
#define IR 2836 
#define NTAB 32 
#define NDIV (1+(IM-1)/NTAB) 
#define EPS 1.2e-14 
#define RNMX (1.0-EPS) 

int ludcmp(double **a, int n, int *indx, double *d)
{
	int i,imax,j,k;
	double big,dum,sum,temp;
	double *vv; /*vv stores the implicit scaling of each row.*/

	vv=dvector(1,n);
	*d=1.0; /* No row interchanges yet. */
	for (i=1;i<=n;i++) { /* Loop over rows to get the implicit scaling information. */
		big=0.0;
		for (j=1;j<=n;j++)
		if ((temp=fabs(a[i][j])) > big) big=temp;
		if (big == 0.0) {
			 /*printf("Singular matrix in routine ludcmp");*/
			return -1;
		}
			 
		/*No nonzero largest element.*/
		vv[i]=1.0/big; /*Save the scaling.*/
	}
	for (j=1;j<=n;j++) { /*This is the loop over columns of Crout's method.*/
		for (i=1;i<j;i++) { /*This is equation (2.3.12) except for i = j.*/
			sum=a[i][j];
			for (k=1;k<i;k++) sum -= a[i][k]*a[k][j];
			a[i][j]=sum;
		}
		big=0.0; /*Initialize for the search for largest pivot element. */
		for (i=j;i<=n;i++) { /* This is i = j of equation (2.3.12) and i = j +1, ..,N */
			sum=a[i][j];     /* of equation (2.3.13). */
			for (k=1;k<j;k++)
				sum -= a[i][k]*a[k][j];
			a[i][j]=sum;
			if ( (dum=vv[i]*fabs(sum)) >= big) {
			/* Is the figure of merit for the pivot better than the best so far? */
				big=dum;
				imax=i;
			}
		}
		if (j != imax) { /* Do we need to interchange rows? */
			for (k=1;k<=n;k++) {  /* Yes, do so... */
				dum=a[imax][k];
				a[imax][k]=a[j][k];
				a[j][k]=dum;
			}
			*d = -(*d); /* ...and change the parity of d. */
			vv[imax]=vv[j]; /* Also interchange the scale factor. */
		}
                 indx[j]=imax;
		if (a[j][j] == 0.0) a[j][j]=TINY;
		if (j != n) { /* Now, finally, divide by the pivot element. */
			dum=1.0/(a[j][j]);
			for (i=j+1;i<=n;i++) a[i][j] *= dum;
		}
	}  /* Go back for the next column in the reduction. */
	free_dvector(vv,1,n);
	
	return 0;
}

void lubksb(double **a, int n, int *indx, double b[])
/* Solves the set of n linear equations A*X = B. Here a[1..n][1..n] is input, not as the matrix
A but rather as its LU decomposition, determined by the routine ludcmp. indx[1..n] is input
as the permutation vector returned by ludcmp. b[1..n] is input as the right-hand side vector
B, and returns with the solution vector X. a, n, and indx are not modiffed by this routine
and can be left in place for successive calls with different right-hand sides b. This routine takes
into account the possibility that b will begin with many zero elements, so it is efficient for use
in matrix inversion. */
{
	int i,ii=0,ip,j;
	double sum;
	for (i=1;i<=n;i++) { 
		ip=indx[i];
		sum=b[ip];
		b[ip]=b[i];
		if (ii)
			for (j=ii;j<=i-1;j++) sum -= a[i][j]*b[j];
		else if (sum) ii=i; /* A nonzero element was encountered, so from now on we
				       will have to do the sums in the loop above. */
		b[i]=sum; 
	}
	for (i=n;i>=1;i--) { /* Now we do the backsubstitution, equation (2.3.7). */
		sum=b[i];
		for (j=i+1;j<=n;j++) sum -= a[i][j]*b[j];
		b[i]=sum/a[i][i]; /* Store a component of the solution vector X. */
	} /* All done! */
}

double ran1(long *idum) 
/* "Minimal" random number generator of Park and Miller with Bays-Durham shuffle 
and added safeguards. Returns a uniform random deviate between 0.0 and 1.0 (exclusive
of the endpoint values). Call with idum a negative integer to initialize; thereafter, 
do not alter idum between successive deviates in a sequence. RNMX should approximate the
largest floating value that is less than 1. */
{ 
    int j; 
    long k; 
    static long iy=0; 
    static long iv[NTAB]; 
    double temp; 
 
    if (*idum <= 0 || !iy) {  /* Initialize. */ 
       if (-(*idum) < 1) *idum=1; /* Be sure to prevent idum = 0. */ 
       else *idum = -(*idum); 
       for (j=NTAB+7;j>=0;j--) {  /* Load the shuffle table (after 8 warm-ups). */ 
               k=(*idum)/IQ; *idum=IA*(*idum-k*IQ)-IR*k; 
               if (*idum < 0) *idum += IM; 
               if (j < NTAB) iv[j] = *idum;
        }
        iy=iv[0]; 
    } 

    k=(*idum)/IQ;   /* Start here when not initializing. */
    *idum=IA*(*idum-k*IQ)-IR*k;    /* Compute idum=(IA*idum) % IM without overflows */ 
    if (*idum < 0) *idum += IM;    /* by  Schrage's method. */ 
    j=iy/NDIV;   /* Will be in the range 0..NTAB-1. */ 
    iy=iv[j];    /* Output previously stored value and refill the shuffle table.*/
    iv[j] = *idum; 
    if ((temp=AM*iy)>RNMX) return RNMX; /* Because users don't expect endpoint values.*/ 
    else return temp; 
}








